import React, { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import styled from 'styled-components';
import { FiClock, FiBarChart, FiBookmark, FiCheck, FiChevronDown, FiChevronUp, FiArrowLeft, FiArrowRight } from 'react-icons/fi';
import { tutorialsAPI } from '../services/api';
import { useUserProgress } from '../contexts/UserProgressContext';
import { Prism as SyntaxHighlighter } from 'react-syntax-highlighter';
import { tomorrow } from 'react-syntax-highlighter/dist/esm/styles/prism';

// Styled Components
const TutorialContainer = styled.div`
  max-width: 1200px;
  margin: 0 auto;
  padding: 1rem 0;
`;

const TutorialHeader = styled.div`
  margin-bottom: 2rem;
  padding-bottom: 1.5rem;
  border-bottom: 1px solid var(--border-color);
`;

const TutorialTitle = styled.h1`
  font-size: 2rem;
  margin-bottom: 0.5rem;
  color: var(--text-color);
`;

const TutorialDescription = styled.p`
  font-size: 1.1rem;
  color: var(--text-light);
  margin-bottom: 1.5rem;
`;

const TutorialMeta = styled.div`
  display: flex;
  align-items: center;
  flex-wrap: wrap;
  gap: 1.5rem;
  margin-bottom: 1.5rem;
  font-size: 0.9rem;
  color: var(--text-light);
`;

const TutorialMetaItem = styled.div`
  display: flex;
  align-items: center;
  gap: 0.5rem;
`;

const CategoryLink = styled(Link)`
  color: var(--primary-color);
  font-weight: 500;
  
  &:hover {
    text-decoration: underline;
  }
`;

const DifficultyBadge = styled.span`
  display: inline-block;
  padding: 0.25rem 0.5rem;
  border-radius: 12px;
  font-size: 0.75rem;
  font-weight: 600;
  text-transform: capitalize;
  background-color: ${props => {
    switch (props.level) {
      case 'beginner':
        return 'rgba(16, 185, 129, 0.1)';
      case 'intermediate':
        return 'rgba(249, 115, 22, 0.1)';
      case 'advanced':
        return 'rgba(239, 68, 68, 0.1)';
      default:
        return 'rgba(16, 185, 129, 0.1)';
    }
  }};
  color: ${props => {
    switch (props.level) {
      case 'beginner':
        return '#10b981';
      case 'intermediate':
        return '#f97316';
      case 'advanced':
        return '#ef4444';
      default:
        return '#10b981';
    }
  }};
`;

const ActionButtons = styled.div`
  display: flex;
  gap: 1rem;
`;

const ActionButton = styled.button`
  display: flex;
  align-items: center;
  gap: 0.5rem;
  padding: 0.5rem 1rem;
  background-color: ${props => props.active ? 'var(--primary-color)' : 'var(--bg-secondary)'};
  color: ${props => props.active ? 'white' : 'var(--text-color)'};
  border: 1px solid ${props => props.active ? 'var(--primary-color)' : 'var(--border-color)'};
  border-radius: var(--radius);
  font-size: 0.9rem;
  font-weight: 500;
  cursor: pointer;
  transition: all var(--transition-speed);
  
  &:hover {
    background-color: ${props => props.active ? 'var(--primary-hover)' : 'var(--border-color)'};
  }
`;

const TutorialContent = styled.div`
  display: flex;
  gap: 2rem;
  
  @media (max-width: 992px) {
    flex-direction: column;
  }
`;

const StepsSidebar = styled.div`
  width: 280px;
  flex-shrink: 0;
  
  @media (max-width: 992px) {
    width: 100%;
  }
`;

const StepsNav = styled.div`
  background-color: var(--bg-color);
  border: 1px solid var(--border-color);
  border-radius: var(--radius);
  overflow: hidden;
  position: sticky;
  top: calc(var(--header-height) + 1rem);
`;

const StepsNavTitle = styled.div`
  padding: 1rem;
  font-weight: 600;
  border-bottom: 1px solid var(--border-color);
  display: flex;
  justify-content: space-between;
  align-items: center;
`;

const StepsList = styled.ul`
  list-style: none;
  padding: 0;
  margin: 0;
  max-height: ${props => props.isOpen ? '500px' : '0'};
  overflow: hidden;
  transition: max-height var(--transition-speed);
  
  @media (min-width: 992px) {
    max-height: none !important;
  }
`;

const StepItem = styled.li`
  border-bottom: 1px solid var(--border-color);
  
  &:last-child {
    border-bottom: none;
  }
`;

const StepLink = styled.button`
  display: flex;
  align-items: center;
  width: 100%;
  text-align: left;
  padding: 0.75rem 1rem;
  background-color: ${props => props.active ? 'rgba(79, 70, 229, 0.1)' : 'transparent'};
  color: ${props => props.active ? 'var(--primary-color)' : 'var(--text-color)'};
  border: none;
  font-size: 0.9rem;
  cursor: pointer;
  transition: all var(--transition-speed);
  
  &:hover {
    background-color: ${props => props.active ? 'rgba(79, 70, 229, 0.15)' : 'rgba(0, 0, 0, 0.05)'};
  }
  
  .step-number {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 24px;
    height: 24px;
    border-radius: 50%;
    background-color: ${props => props.completed ? 'var(--primary-color)' : props.active ? 'rgba(79, 70, 229, 0.2)' : 'var(--bg-secondary)'};
    color: ${props => props.completed ? 'white' : props.active ? 'var(--primary-color)' : 'var(--text-light)'};
    margin-right: 0.75rem;
    font-size: 0.8rem;
    font-weight: 600;
  }
`;

const MainContent = styled.div`
  flex: 1;
`;

const StepContent = styled.div`
  margin-bottom: 3rem;
`;

const StepHeader = styled.div`
  margin-bottom: 1.5rem;
`;

const StepTitle = styled.h2`
  font-size: 1.75rem;
  margin-bottom: 1rem;
  color: var(--text-color);
  display: flex;
  align-items: center;
  gap: 0.75rem;
  
  .step-number {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background-color: var(--primary-color);
    color: white;
    font-size: 0.9rem;
    font-weight: 600;
  }
`;

const StepDescription = styled.div`
  font-size: 1.1rem;
  line-height: 1.6;
  color: var(--text-color);
  margin-bottom: 2rem;
`;

const ExplanationTabs = styled.div`
  display: flex;
  border-bottom: 1px solid var(--border-color);
  margin-bottom: 1.5rem;
`;

const ExplanationTab = styled.button`
  padding: 0.75rem 1.5rem;
  background: none;
  border: none;
  border-bottom: 2px solid ${props => props.active ? 'var(--primary-color)' : 'transparent'};
  color: ${props => props.active ? 'var(--primary-color)' : 'var(--text-light)'};
  font-weight: ${props => props.active ? '600' : '400'};
  cursor: pointer;
  transition: all var(--transition-speed);
  
  &:hover {
    color: var(--primary-color);
  }
`;

const ExplanationContent = styled.div`
  margin-bottom: 2rem;
  font-size: 1rem;
  line-height: 1.6;
  color: var(--text-color);
`;

const CodeBlock = styled.div`
  margin: 1.5rem 0;
  border-radius: var(--radius);
  overflow: hidden;
`;

const NavigationButtons = styled.div`
  display: flex;
  justify-content: space-between;
  margin-top: 3rem;
  padding-top: 1.5rem;
  border-top: 1px solid var(--border-color);
`;

const NavButton = styled(Link)`
  display: flex;
  align-items: center;
  gap: 0.5rem;
  padding: 0.75rem 1.5rem;
  background-color: var(--bg-secondary);
  color: var(--text-color);
  border-radius: var(--radius);
  font-weight: 500;
  transition: all var(--transition-speed);
  
  &:hover {
    background-color: var(--border-color);
  }
  
  &.next {
    background-color: var(--primary-color);
    color: white;
    
    &:hover {
      background-color: var(--primary-hover);
    }
  }
  
  &.disabled {
    opacity: 0.5;
    pointer-events: none;
  }
`;

const CompleteButton = styled.button`
  display: flex;
  align-items: center;
  justify-content: center;
  gap: 0.5rem;
  padding: 0.75rem 1.5rem;
  background-color: ${props => props.completed ? 'var(--success-color)' : 'var(--primary-color)'};
  color: white;
  border: none;
  border-radius: var(--radius);
  font-weight: 600;
  cursor: pointer;
  transition: all var(--transition-speed);
  width: 100%;
  margin-top: 1.5rem;
  
  &:hover {
    background-color: ${props => props.completed ? 'var(--success-hover)' : 'var(--primary-hover)'};
  }
`;

const ProgressBar = styled.div`
  width: 100%;
  height: 6px;
  background-color: var(--bg-secondary);
  border-radius: 3px;
  overflow: hidden;
  margin-top: 1rem;
  
  .progress {
    height: 100%;
    background-color: var(--primary-color);
    width: ${props => props.progress}%;
    transition: width var(--transition-speed);
  }
`;

const LoadingState = styled.div`
  display: flex;
  justify-content: center;
  align-items: center;
  padding: 3rem 0;
  
  .spinner {
    width: 40px;
    height: 40px;
    border: 3px solid rgba(0, 0, 0, 0.1);
    border-radius: 50%;
    border-top-color: var(--primary-color);
    animation: spin 1s ease-in-out infinite;
  }
  
  @keyframes spin {
    to { transform: rotate(360deg); }
  }
`;

const EmptyState = styled.div`
  text-align: center;
  padding: 3rem 1rem;
  background-color: var(--bg-secondary);
  border-radius: var(--radius);
  
  h3 {
    margin-bottom: 1rem;
  }
  
  p {
    color: var(--text-light);
    margin-bottom: 1.5rem;
  }
`;

const TutorialPage = () => {
  const { slug } = useParams();
  const [tutorial, setTutorial] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [currentStep, setCurrentStep] = useState(0);
  const [explanationMode, setExplanationMode] = useState('beginner');
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const { 
    markStepComplete, 
    isStepCompleted, 
    saveTutorial, 
    isTutorialSaved, 
    getTutorialProgress 
  } = useUserProgress();
  
  useEffect(() => {
    const fetchTutorialData = async () => {
      setLoading(true);
      try {
        const response = await tutorialsAPI.getTutorialBySlug(slug);
        if (response.status === 'success') {
          setTutorial(response.data);
        } else {
          setError('Failed to load tutorial data');
        }
      } catch (err) {
        console.error('Error fetching tutorial:', err);
        setError('Failed to load tutorial. Please try again later.');
      } finally {
        setLoading(false);
      }
    };
    
    fetchTutorialData();
  }, [slug]);
  
  const handleStepChange = (index) => {
    setCurrentStep(index);
    // Scroll to top when changing steps
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };
  
  const handleCompleteStep = () => {
    if (tutorial) {
      markStepComplete(tutorial.id, currentStep);
    }
  };
  
  const handleSaveTutorial = () => {
    if (tutorial) {
      saveTutorial(tutorial);
    }
  };
  
  const toggleSidebar = () => {
    setSidebarOpen(!sidebarOpen);
  };
  
  const getProgress = () => {
    if (!tutorial) return 0;
    return getTutorialProgress(tutorial.id, tutorial.steps.length);
  };
  
  const getNextStepIndex = () => {
    if (!tutorial || currentStep >= tutorial.steps.length - 1) return null;
    return currentStep + 1;
  };
  
  const getPrevStepIndex = () => {
    if (!tutorial || currentStep <= 0) return null;
    return currentStep - 1;
  };
  
  if (loading) {
    return (
      <TutorialContainer>
        <LoadingState>
          <div className="spinner"></div>
        </LoadingState>
      </TutorialContainer>
    );
  }
  
  if (error || !tutorial) {
    return (
      <TutorialContainer>
        <EmptyState>
          <h3>Error Loading Tutorial</h3>
          <p>{error || 'Tutorial not found'}</p>
          <Link to="/" className="btn btn-primary">
            Return to Home
          </Link>
        </EmptyState>
      </TutorialContainer>
    );
  }
  
  const currentStepData = tutorial.steps[currentStep];
  
  return (
    <TutorialContainer>
      <TutorialHeader>
        <TutorialTitle>{tutorial.title}</TutorialTitle>
        <TutorialDescription>{tutorial.description}</TutorialDescription>
        <TutorialMeta>
          <TutorialMetaItem>
            <DifficultyBadge level={tutorial.difficulty_level}>
              {tutorial.difficulty_level}
            </DifficultyBadge>
          </TutorialMetaItem>
          <TutorialMetaItem>
            <FiClock /> {tutorial.estimated_time} min
          </TutorialMetaItem>
          <TutorialMetaItem>
            <FiBarChart /> {tutorial.views} views
          </TutorialMetaItem>
          <TutorialMetaItem>
            Category: <CategoryLink to={`/category/${tutorial.category_slug}`}>
              {tutorial.category_name}
            </CategoryLink>
          </TutorialMetaItem>
        </TutorialMeta>
        <ActionButtons>
          <ActionButton 
            onClick={handleSaveTutorial}
            active={isTutorialSaved(tutorial.id)}
          >
            <FiBookmark /> {isTutorialSaved(tutorial.id) ? 'Saved' : 'Save for Later'}
          </ActionButton>
        </ActionButtons>
      </TutorialHeader>
      
      <TutorialContent>
        <StepsSidebar>
          <StepsNav>
            <StepsNavTitle>
              Tutorial Steps
              <button onClick={toggleSidebar}>
                {sidebarOpen ? <FiChevronUp /> : <FiChevronDown />}
              </button>
            </StepsNavTitle>
            <StepsList isOpen={sidebarOpen}>
              {tutorial.steps.map((step, index) => (
                <StepItem key={index}>
                  <StepLink 
                    onClick={() => handleStepChange(index)}
                    active={currentStep === index}
                    completed={isStepCompleted(tutorial.id, index)}
                  >
                    <span className="step-number">
                      {isStepCompleted(tutorial.id, index) ? <FiCheck /> : index + 1}
                    </span>
                    {step.title}
                  </StepLink>
                </StepItem>
              ))}
            </StepsList>
            <div style={{ padding: '1rem' }}>
              <ProgressBar progress={getProgress()}>
                <div className="progress"></div>
              </ProgressBar>
              <CompleteButton 
                onClick={handleCompleteStep}
                completed={isStepCompleted(tutorial.id, currentStep)}
              >
                {isStepCompleted(tutorial.id, currentStep) ? (
                  <>
                    <FiCheck /> Completed
                  </>
                ) : (
                  'Mark as Complete'
                )}
              </CompleteButton>
            </div>
          </StepsNav>
        </StepsSidebar>
        
        <MainContent>
          <StepContent>
            <StepHeader>
              <StepTitle>
                <span className="step-number">{currentStep + 1}</span>
                {currentStepData.title}
              </StepTitle>
            </StepHeader>
            
            <StepDescription>
              {currentStepData.content}
            </StepDescription>
            
            {(currentStepData.beginner_content || currentStepData.advanced_content) && (
              <>
                <ExplanationTabs>
                  <ExplanationTab 
                    active={explanationMode === 'beginner'}
                    onClick={() => setExplanationMode('beginner')}
                  >
                    Beginner Explanation
                  </ExplanationTab>
                  <ExplanationTab 
                    active={explanationMode === 'advanced'}
                    onClick={() => setExplanationMode('advanced')}
                  >
                    Advanced Explanation
                  </ExplanationTab>
                </ExplanationTabs>
                
                <ExplanationContent>
                  {explanationMode === 'beginner' ? 
                    currentStepData.beginner_content : 
                    currentStepData.advanced_content
                  }
                </ExplanationContent>
              </>
            )}
            
            {currentStepData.code_snippet && (
              <CodeBlock>
                <SyntaxHighlighter 
                  language={currentStepData.code_language || 'javascript'} 
                  style={tomorrow}
                  showLineNumbers={true}
                >
                  {currentStepData.code_snippet}
                </SyntaxHighlighter>
              </CodeBlock>
            )}
            
            <NavigationButtons>
              {getPrevStepIndex() !== null ? (
                <NavButton onClick={() => handleStepChange(getPrevStepIndex())}>
                  <FiArrowLeft /> Previous Step
                </NavButton>
              ) : (
                <div></div>
              )}
              
              {getNextStepIndex() !== null ? (
                <NavButton 
                  className="next"
                  onClick={() => handleStepChange(getNextStepIndex())}
                >
                  Next Step <FiArrowRight />
                </NavButton>
              ) : (
                <NavButton 
                  to={`/category/${tutorial.category_slug}`}
                  className="next"
                >
                  Finish Tutorial <FiCheck />
                </NavButton>
              )}
            </NavigationButtons>
          </StepContent>
        </MainContent>
      </TutorialContent>
    </TutorialContainer>
  );
};

export default TutorialPage;