import React, { useState, useEffect } from "react";
import { useLocation, Link } from "react-router-dom";
import styled from "styled-components";
import { FiSearch, FiClock, FiBarChart, FiGrid, FiList } from "react-icons/fi";
import { tutorialsAPI } from "../services/api";

// Styled Components
const SearchContainer = styled.div`
  max-width: 1200px;
  margin: 0 auto;
  padding: 1rem 0;
`;

const SearchHeader = styled.div`
  margin-bottom: 2rem;
  padding-bottom: 1.5rem;
  border-bottom: 1px solid var(--border-color);
`;

const SearchTitle = styled.h1`
  font-size: 2rem;
  margin-bottom: 0.5rem;
  color: var(--text-color);
`;

const SearchDescription = styled.p`
  font-size: 1.1rem;
  color: var(--text-light);
  margin-bottom: 1.5rem;
`;

const SearchForm = styled.form`
  display: flex;
  gap: 0.5rem;
  margin-bottom: 1.5rem;
`;

const SearchInput = styled.input`
  flex: 1;
  padding: 0.75rem 1rem;
  border: 1px solid var(--border-color);
  border-radius: var(--radius);
  background-color: var(--bg-secondary);
  color: var(--text-color);
  font-size: 1rem;

  &:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 2px rgba(79, 70, 229, 0.2);
  }
`;

const SearchButton = styled.button`
  padding: 0.75rem 1.5rem;
  background-color: var(--primary-color);
  color: white;
  border: none;
  border-radius: var(--radius);
  font-weight: 600;
  cursor: pointer;
  display: flex;
  align-items: center;
  gap: 0.5rem;

  &:hover {
    background-color: var(--primary-hover);
  }
`;

const ResultsHeader = styled.div`
  display: flex;
  justify-content: space-between;
  align-items: center;
  margin-bottom: 1.5rem;
`;

const ResultsCount = styled.div`
  font-size: 1rem;
  color: var(--text-light);
`;

const ViewOptions = styled.div`
  display: flex;
  gap: 0.5rem;
`;

const ViewButton = styled.button`
  background-color: ${(props) =>
    props.active ? "var(--primary-color)" : "var(--bg-secondary)"};
  color: ${(props) => (props.active ? "white" : "var(--text-color)")};
  border: none;
  border-radius: var(--radius);
  padding: 0.5rem;
  cursor: pointer;
  transition: all var(--transition-speed);
  display: flex;
  align-items: center;
  justify-content: center;

  &:hover {
    background-color: ${(props) =>
      props.active ? "var(--primary-hover)" : "var(--border-color)"};
  }
`;

const ResultsGrid = styled.div`
  display: grid;
  grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
  gap: 1.5rem;
  margin-bottom: 1.5rem;
`;

const ResultsList = styled.div`
  display: flex;
  flex-direction: column;
  gap: 1rem;
  margin-bottom: 1.5rem;
`;

const ResultCard = styled(Link)`
  display: flex;
  flex-direction: column;
  background-color: var(--bg-color);
  border: 1px solid var(--border-color);
  border-radius: var(--radius);
  overflow: hidden;
  box-shadow: var(--box-shadow);
  transition: all var(--transition-speed);
  color: var(--text-color);

  &:hover {
    transform: translateY(-5px);
    box-shadow: var(--box-shadow-md);
    border-color: var(--primary-color);
  }
`;

const ResultListItem = styled(Link)`
  display: flex;
  gap: 1.5rem;
  padding: 1.5rem;
  background-color: var(--bg-color);
  border: 1px solid var(--border-color);
  border-radius: var(--radius);
  box-shadow: var(--box-shadow);
  transition: all var(--transition-speed);
  color: var(--text-color);

  &:hover {
    transform: translateY(-3px);
    box-shadow: var(--box-shadow-md);
    border-color: var(--primary-color);
  }

  @media (max-width: 768px) {
    flex-direction: column;
    gap: 1rem;
  }
`;

const ResultContent = styled.div`
  padding: 1.5rem;
  flex: 1;
  display: flex;
  flex-direction: column;
`;

const ResultListContent = styled.div`
  flex: 1;
  display: flex;
  flex-direction: column;
`;

const ResultTitle = styled.h3`
  font-size: 1.1rem;
  margin-bottom: 0.5rem;
`;

const ResultDescription = styled.p`
  font-size: 0.9rem;
  color: var(--text-light);
  margin-bottom: 1rem;
  flex: 1;
`;

const ResultMeta = styled.div`
  display: flex;
  align-items: center;
  gap: 1rem;
  font-size: 0.8rem;
  color: var(--text-light);
`;

const ResultMetaItem = styled.div`
  display: flex;
  align-items: center;
  gap: 0.25rem;
`;

const CategoryBadge = styled.span`
  display: inline-block;
  padding: 0.25rem 0.5rem;
  border-radius: 12px;
  font-size: 0.75rem;
  background-color: rgba(79, 70, 229, 0.1);
  color: var(--primary-color);
  margin-bottom: 1rem;
`;

const DifficultyBadge = styled.span`
  display: inline-block;
  padding: 0.25rem 0.5rem;
  border-radius: 12px;
  font-size: 0.75rem;
  font-weight: 600;
  text-transform: capitalize;
  background-color: ${(props) => {
    switch (props.level) {
      case "beginner":
        return "rgba(16, 185, 129, 0.1)";
      case "intermediate":
        return "rgba(249, 115, 22, 0.1)";
      case "advanced":
        return "rgba(239, 68, 68, 0.1)";
      default:
        return "rgba(16, 185, 129, 0.1)";
    }
  }};
  color: ${(props) => {
    switch (props.level) {
      case "beginner":
        return "#10b981";
      case "intermediate":
        return "#f97316";
      case "advanced":
        return "#ef4444";
      default:
        return "#10b981";
    }
  }};
  margin-right: 0.5rem;
`;

const LoadingState = styled.div`
  display: flex;
  justify-content: center;
  align-items: center;
  padding: 3rem 0;

  .spinner {
    width: 40px;
    height: 40px;
    border: 3px solid rgba(0, 0, 0, 0.1);
    border-radius: 50%;
    border-top-color: var(--primary-color);
    animation: spin 1s ease-in-out infinite;
  }

  @keyframes spin {
    to {
      transform: rotate(360deg);
    }
  }
`;

const EmptyState = styled.div`
  text-align: center;
  padding: 3rem 1rem;
  background-color: var(--bg-secondary);
  border-radius: var(--radius);

  h3 {
    margin-bottom: 1rem;
  }

  p {
    color: var(--text-light);
    margin-bottom: 1.5rem;
  }
`;

const SearchResultsPage = () => {
  const location = useLocation();
  const queryParams = new URLSearchParams(location.search);
  const initialQuery = queryParams.get("q") || "";

  const [searchQuery, setSearchQuery] = useState(initialQuery);
  const [currentSearchQuery, setCurrentSearchQuery] = useState(initialQuery);
  const [searchResults, setSearchResults] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [viewMode, setViewMode] = useState("grid");
  const [searchSource, setSearchSource] = useState(null);

  useEffect(() => {
    if (initialQuery) {
      performSearch(initialQuery);
    }
  }, [initialQuery]);

  const performSearch = async (query) => {
    if (!query.trim()) return;

    setLoading(true);
    setError(null);

    try {
      const response = await tutorialsAPI.searchTutorials(query);

      console.log(response);

      // Check if the response contains an error field
      if (response.error) {
        let errorMessage;
        let searchTips = "";

        if (response.error === "Failed to generate tutorial steps") {
          errorMessage =
            "We couldn't generate tutorial results for your search.";
          searchTips =
            "Try using more specific keywords or check your spelling.";
        } else if (response.error === "Failed to generate category content") {
          errorMessage = "We couldn't categorize your search request.";
          searchTips =
            "Try using more common terms or be more specific about what you're looking for.";
        } else {
          errorMessage = response.error;
        }

        // Combine the error message with search tips
        setError(`${errorMessage} ${searchTips}`);
        setSearchSource(response.source);
        setSearchResults([]);
      } else if (response.source === "database" || response.source === "ai") {
        setSearchResults(response.results || []);
        setSearchSource(response.source);
      } else {
        setError("Invalid response format from search API");
        setSearchResults([]);
      }
    } catch (err) {
      console.error("Error searching tutorials:", err);
      setError("Failed to search tutorials. Please try again later.");
      setSearchResults([]);
    } finally {
      setLoading(false);
    }
  };

  const handleSearchSubmit = (e) => {
    e.preventDefault();

    // Update the URL
    const newUrl = `/search?q=${encodeURIComponent(searchQuery)}`;
    window.history.pushState({}, "", newUrl);

    // Update the current search query for display
    setCurrentSearchQuery(searchQuery);

    // Update the page title
    document.title = `Search: ${searchQuery} | Your Site Name`;

    performSearch(searchQuery);
  };

  return (
    <SearchContainer>
      <SearchHeader>
        <SearchTitle>Search Results</SearchTitle>
        <SearchDescription>
          {currentSearchQuery
            ? `Showing results for "${currentSearchQuery}"`
            : "Search for tutorials"}
        </SearchDescription>

        <SearchForm onSubmit={handleSearchSubmit}>
          <SearchInput
            type="text"
            placeholder="Search tutorials..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
          />
          <SearchButton type="submit">
            <FiSearch /> Search
          </SearchButton>
        </SearchForm>
      </SearchHeader>

      {loading ? (
        <LoadingState>
          <div className="spinner"></div>
        </LoadingState>
      ) : error ? (
        <EmptyState>
          <h3>Error</h3>
          <p>{error}</p>
        </EmptyState>
      ) : searchResults.length > 0 ? (
        <>
          <ResultsHeader>
            <ResultsCount>
              {searchResults.length}{" "}
              {searchResults.length === 1 ? "result" : "results"} found
              {searchSource === "ai" && " (AI-generated)"}
            </ResultsCount>
            <ViewOptions>
              <ViewButton
                active={viewMode === "grid"}
                onClick={() => setViewMode("grid")}
                title="Grid View">
                <FiGrid />
              </ViewButton>
              <ViewButton
                active={viewMode === "list"}
                onClick={() => setViewMode("list")}
                title="List View">
                <FiList />
              </ViewButton>
            </ViewOptions>
          </ResultsHeader>

          {viewMode === "grid" ? (
            <ResultsGrid>
              {searchResults.map((result) => {
                // Extract the tutorial data from the nested structure
                const tutorialData = result.tutorial || result;

                return (
                  <ResultCard
                    key={tutorialData.id}
                    to={`/tutorial/${tutorialData.slug || ""}`}>
                    <ResultContent>
                      <CategoryBadge>
                        {tutorialData.category_name}
                      </CategoryBadge>
                      <ResultTitle>{tutorialData.title}</ResultTitle>
                      <ResultDescription>
                        {tutorialData.description}
                      </ResultDescription>
                      <ResultMeta>
                        <ResultMetaItem>
                          <DifficultyBadge
                            level={tutorialData.difficulty_level}>
                            {tutorialData.difficulty_level}
                          </DifficultyBadge>
                        </ResultMetaItem>
                        <ResultMetaItem>
                          <FiClock /> {tutorialData.estimated_time} min
                        </ResultMetaItem>
                      </ResultMeta>
                    </ResultContent>
                  </ResultCard>
                );
              })}
            </ResultsGrid>
          ) : (
            <ResultsList>
              {searchResults.map((result) => {
                // Extract the tutorial data from the nested structure
                const tutorialData = result.tutorial || result;

                return (
                  <ResultCard
                    key={tutorialData.id}
                    to={`/tutorial/${tutorialData.slug || ""}`}>
                    <ResultContent>
                      <CategoryBadge>
                        {tutorialData.category_name}
                      </CategoryBadge>
                      <ResultTitle>{tutorialData.title}</ResultTitle>
                      <ResultDescription>
                        {tutorialData.description}
                      </ResultDescription>
                      <ResultMeta>
                        <ResultMetaItem>
                          <DifficultyBadge
                            level={tutorialData.difficulty_level}>
                            {tutorialData.difficulty_level}
                          </DifficultyBadge>
                        </ResultMetaItem>
                        <ResultMetaItem>
                          <FiClock /> {tutorialData.estimated_time} min
                        </ResultMetaItem>
                      </ResultMeta>
                    </ResultContent>
                  </ResultCard>
                );
              })}
            </ResultsList>
          )}
        </>
      ) : currentSearchQuery ? (
        <EmptyState>
          <h3>No Results Found</h3>
          <p>
            No tutorials found for "{currentSearchQuery}". Try a different
            search term or browse categories.
          </p>
          <Link to="/" className="btn btn-primary">
            Browse Categories
          </Link>
        </EmptyState>
      ) : (
        <EmptyState>
          <h3>Search for Tutorials</h3>
          <p>Enter a search term above to find tutorials.</p>
        </EmptyState>
      )}
    </SearchContainer>
  );
};

export default SearchResultsPage;
