import React, { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import styled from 'styled-components';
import { FiArrowRight, FiSearch, FiBookOpen, FiClock, FiBarChart } from 'react-icons/fi';
import { tutorialsAPI } from '../services/api';

// Styled Components
const HomeContainer = styled.div`
  max-width: 1200px;
  margin: 0 auto;
  padding: 2rem 1rem;
`;

const HeroSection = styled.section`
  text-align: center;
  padding: 3rem 1rem;
  margin-bottom: 3rem;
  background-color: var(--bg-secondary);
  border-radius: var(--radius);
  box-shadow: var(--box-shadow);
`;

const HeroTitle = styled.h1`
  font-size: 2.5rem;
  margin-bottom: 1rem;
  color: var(--text-color);
  
  @media (max-width: 768px) {
    font-size: 2rem;
  }
`;

const HeroSubtitle = styled.p`
  font-size: 1.25rem;
  color: var(--text-light);
  margin-bottom: 2rem;
  max-width: 700px;
  margin-left: auto;
  margin-right: auto;
  
  @media (max-width: 768px) {
    font-size: 1rem;
  }
`;

const SearchForm = styled.form`
  display: flex;
  max-width: 600px;
  margin: 0 auto;
  
  @media (max-width: 576px) {
    flex-direction: column;
  }
`;

const SearchInput = styled.input`
  flex: 1;
  padding: 0.75rem 1rem;
  border: 2px solid var(--border-color);
  border-radius: var(--radius) 0 0 var(--radius);
  font-size: 1rem;
  background-color: var(--bg-color);
  color: var(--text-color);
  
  &:focus {
    outline: none;
    border-color: var(--primary-color);
  }
  
  @media (max-width: 576px) {
    border-radius: var(--radius);
    margin-bottom: 0.5rem;
  }
`;

const SearchButton = styled.button`
  padding: 0.75rem 1.5rem;
  background-color: var(--primary-color);
  color: white;
  border: none;
  border-radius: 0 var(--radius) var(--radius) 0;
  font-size: 1rem;
  font-weight: 600;
  cursor: pointer;
  display: flex;
  align-items: center;
  gap: 0.5rem;
  transition: background-color var(--transition-speed);
  
  &:hover {
    background-color: var(--primary-hover);
  }
  
  @media (max-width: 576px) {
    border-radius: var(--radius);
    justify-content: center;
  }
`;

const SectionTitle = styled.h2`
  font-size: 1.75rem;
  margin-bottom: 1.5rem;
  color: var(--text-color);
  display: flex;
  align-items: center;
  gap: 0.75rem;
  
  svg {
    color: var(--primary-color);
  }
`;

const CategoryGrid = styled.div`
  display: grid;
  grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
  gap: 1.5rem;
  margin-bottom: 3rem;
`;

const CategoryCard = styled(Link)`
  display: flex;
  flex-direction: column;
  padding: 1.5rem;
  background-color: var(--bg-color);
  border: 1px solid var(--border-color);
  border-radius: var(--radius);
  box-shadow: var(--box-shadow);
  transition: all var(--transition-speed);
  color: var(--text-color);
  
  &:hover {
    transform: translateY(-5px);
    box-shadow: var(--box-shadow-md);
    border-color: var(--primary-color);
  }
`;

const CategoryIcon = styled.div`
  width: 50px;
  height: 50px;
  background-color: rgba(79, 70, 229, 0.1);
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
  margin-bottom: 1rem;
  color: var(--primary-color);
  font-size: 1.5rem;
`;

const CategoryName = styled.h3`
  font-size: 1.25rem;
  margin-bottom: 0.5rem;
`;

const CategoryCount = styled.p`
  font-size: 0.9rem;
  color: var(--text-light);
  margin-bottom: 0.5rem;
`;

const CategoryDescription = styled.p`
  font-size: 0.9rem;
  color: var(--text-light);
  margin-bottom: 1rem;
  flex: 1;
`;

const TutorialGrid = styled.div`
  display: grid;
  grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
  gap: 1.5rem;
  margin-bottom: 3rem;
`;

const TutorialCard = styled(Link)`
  display: flex;
  flex-direction: column;
  background-color: var(--bg-color);
  border: 1px solid var(--border-color);
  border-radius: var(--radius);
  overflow: hidden;
  box-shadow: var(--box-shadow);
  transition: all var(--transition-speed);
  color: var(--text-color);
  
  &:hover {
    transform: translateY(-5px);
    box-shadow: var(--box-shadow-md);
    border-color: var(--primary-color);
  }
`;

const TutorialContent = styled.div`
  padding: 1.5rem;
  flex: 1;
  display: flex;
  flex-direction: column;
`;

const TutorialTitle = styled.h3`
  font-size: 1.1rem;
  margin-bottom: 0.5rem;
`;

const TutorialDescription = styled.p`
  font-size: 0.9rem;
  color: var(--text-light);
  margin-bottom: 1rem;
  flex: 1;
`;

const TutorialMeta = styled.div`
  display: flex;
  align-items: center;
  gap: 1rem;
  font-size: 0.8rem;
  color: var(--text-light);
`;

const TutorialMetaItem = styled.div`
  display: flex;
  align-items: center;
  gap: 0.25rem;
`;

const DifficultyBadge = styled.span`
  display: inline-block;
  padding: 0.25rem 0.5rem;
  border-radius: 12px;
  font-size: 0.75rem;
  font-weight: 600;
  text-transform: capitalize;
  background-color: ${props => {
    switch (props.level) {
      case 'beginner':
        return 'rgba(16, 185, 129, 0.1)';
      case 'intermediate':
        return 'rgba(249, 115, 22, 0.1)';
      case 'advanced':
        return 'rgba(239, 68, 68, 0.1)';
      default:
        return 'rgba(16, 185, 129, 0.1)';
    }
  }};
  color: ${props => {
    switch (props.level) {
      case 'beginner':
        return '#10b981';
      case 'intermediate':
        return '#f97316';
      case 'advanced':
        return '#ef4444';
      default:
        return '#10b981';
    }
  }};
  margin-bottom: 1rem;
`;

const ViewAllLink = styled(Link)`
  display: flex;
  align-items: center;
  justify-content: center;
  gap: 0.5rem;
  margin-top: 1rem;
  padding: 0.75rem 1.5rem;
  background-color: var(--bg-secondary);
  color: var(--primary-color);
  border-radius: var(--radius);
  font-weight: 600;
  transition: all var(--transition-speed);
  
  &:hover {
    background-color: rgba(79, 70, 229, 0.1);
  }
`;

const FeatureSection = styled.section`
  margin-bottom: 3rem;
  padding: 2rem;
  background-color: var(--bg-secondary);
  border-radius: var(--radius);
`;

const FeatureGrid = styled.div`
  display: grid;
  grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
  gap: 2rem;
`;

const FeatureItem = styled.div`
  display: flex;
  flex-direction: column;
  align-items: center;
  text-align: center;
`;

const FeatureIcon = styled.div`
  width: 60px;
  height: 60px;
  background-color: rgba(79, 70, 229, 0.1);
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
  margin-bottom: 1rem;
  color: var(--primary-color);
  font-size: 1.75rem;
`;

const FeatureTitle = styled.h3`
  font-size: 1.25rem;
  margin-bottom: 0.75rem;
`;

const FeatureDescription = styled.p`
  font-size: 0.9rem;
  color: var(--text-light);
`;

const HomePage = () => {
  const [searchQuery, setSearchQuery] = useState('');
  const [popularTutorials, setPopularTutorials] = useState([]);
  const [categories, setCategories] = useState([]);
  const [loading, setLoading] = useState(true);
  
  useEffect(() => {
    const fetchHomeData = async () => {
      try {
        // Fetch popular tutorials
        const tutorialsResponse = await tutorialsAPI.getAllTutorials({
          limit: 6,
          sort_by: 'views',
          sort_order: 'desc'
        });
        
        if (tutorialsResponse.status === 'success') {
          setPopularTutorials(tutorialsResponse.data);
        }
        
        // Fetch categories
        const categoriesResponse = await fetch('/api/categories');
        const categoriesData = await categoriesResponse.json();
        
        if (categoriesData.status === 'success') {
          setCategories(categoriesData.data);
        }
      } catch (error) {
        console.error('Error fetching home data:', error);
      } finally {
        setLoading(false);
      }
    };
    
    fetchHomeData();
  }, []);
  
  const handleSearchSubmit = (e) => {
    e.preventDefault();
    if (searchQuery.trim()) {
      window.location.href = `/search?q=${encodeURIComponent(searchQuery.trim())}`;
    }
  };
  
  // Helper function to get appropriate icon for a category
  const getCategoryIcon = (categoryName) => {
    const name = categoryName.toLowerCase();
    if (name.includes('frontend') || name.includes('react')) {
      return <FiBookOpen />;
    } else if (name.includes('backend') || name.includes('node')) {
      return <FiBarChart />;
    } else {
      return <FiBookOpen />;
    }
  };
  
  return (
    <HomeContainer>
      <HeroSection>
        <HeroTitle>Learn with AI-Powered Tutorials</HeroTitle>
        <HeroSubtitle>
          Discover step-by-step tutorials on programming, web development, and more.
          Our AI-powered platform adapts content to your learning style.
        </HeroSubtitle>
        
        <SearchForm onSubmit={handleSearchSubmit}>
          <SearchInput
            type="text"
            placeholder="What do you want to learn today?"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
          />
          <SearchButton type="submit">
            <FiSearch /> Search
          </SearchButton>
        </SearchForm>
      </HeroSection>
      
      <section>
        <SectionTitle>
          <FiBookOpen /> Popular Categories
        </SectionTitle>
        
        <CategoryGrid>
          {categories.slice(0, 6).map(category => (
            <CategoryCard key={category.id} to={`/category/${category.slug}`}>
              <CategoryIcon>
                {getCategoryIcon(category.name)}
              </CategoryIcon>
              <CategoryName>{category.name}</CategoryName>
              <CategoryCount>{category.tutorial_count} tutorials</CategoryCount>
              <CategoryDescription>
                {category.description || `Explore ${category.name} tutorials and guides.`}
              </CategoryDescription>
            </CategoryCard>
          ))}
        </CategoryGrid>
        
        <ViewAllLink to="/categories">
          View all categories <FiArrowRight />
        </ViewAllLink>
      </section>
      
      <section>
        <SectionTitle>
          <FiBarChart /> Popular Tutorials
        </SectionTitle>
        
        <TutorialGrid>
          {popularTutorials.map(tutorial => (
            <TutorialCard key={tutorial.id} to={`/tutorial/${tutorial.slug}`}>
              <TutorialContent>
                <DifficultyBadge level={tutorial.difficulty_level}>
                  {tutorial.difficulty_level}
                </DifficultyBadge>
                <TutorialTitle>{tutorial.title}</TutorialTitle>
                <TutorialDescription>
                  {tutorial.description}
                </TutorialDescription>
                <TutorialMeta>
                  <TutorialMetaItem>
                    <FiClock /> {tutorial.estimated_time} min
                  </TutorialMetaItem>
                  <TutorialMetaItem>
                    {tutorial.category_name}
                  </TutorialMetaItem>
                </TutorialMeta>
              </TutorialContent>
            </TutorialCard>
          ))}
        </TutorialGrid>
        
        <ViewAllLink to="/tutorials">
          View all tutorials <FiArrowRight />
        </ViewAllLink>
      </section>
      
      <FeatureSection>
        <SectionTitle>Why TutorAI?</SectionTitle>
        
        <FeatureGrid>
          <FeatureItem>
            <FeatureIcon>
              <FiBookOpen />
            </FeatureIcon>
            <FeatureTitle>Adaptive Learning</FeatureTitle>
            <FeatureDescription>
              Content adapts to your skill level, with beginner and advanced explanations for each step.
            </FeatureDescription>
          </FeatureItem>
          
          <FeatureItem>
            <FeatureIcon>
              <FiSearch />
            </FeatureIcon>
            <FeatureTitle>AI-Powered Search</FeatureTitle>
            <FeatureDescription>
              Can't find what you need? Our AI will create a custom tutorial just for you.
            </FeatureDescription>
          </FeatureItem>
          
          <FeatureItem>
            <FeatureIcon>
              <FiClock />
            </FeatureIcon>
            <FeatureTitle>Track Progress</FeatureTitle>
            <FeatureDescription>
              Mark steps as complete and save tutorials for later to keep track of your learning journey.
            </FeatureDescription>
          </FeatureItem>
        </FeatureGrid>
      </FeatureSection>
    </HomeContainer>
  );
};

export default HomePage;