import React, { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import styled from 'styled-components';
import { FiArrowRight, FiBookOpen, FiClock, FiBarChart, FiGrid, FiList } from 'react-icons/fi';
import { categoriesAPI } from '../services/api';

// Styled Components
const CategoryContainer = styled.div`
  max-width: 1200px;
  margin: 0 auto;
  padding: 1rem 0;
`;

const CategoryHeader = styled.div`
  margin-bottom: 2rem;
  padding-bottom: 1.5rem;
  border-bottom: 1px solid var(--border-color);
`;

const CategoryTitle = styled.h1`
  font-size: 2rem;
  margin-bottom: 0.5rem;
  color: var(--text-color);
`;

const CategoryDescription = styled.p`
  font-size: 1.1rem;
  color: var(--text-light);
  margin-bottom: 1.5rem;
`;

const CategoryStats = styled.div`
  display: flex;
  align-items: center;
  gap: 1.5rem;
  font-size: 0.9rem;
  color: var(--text-light);
`;

const CategoryStat = styled.div`
  display: flex;
  align-items: center;
  gap: 0.5rem;
`;

const SubcategorySection = styled.section`
  margin-bottom: 2rem;
`;

const SectionTitle = styled.h2`
  font-size: 1.5rem;
  margin-bottom: 1.5rem;
  color: var(--text-color);
  display: flex;
  align-items: center;
  gap: 0.75rem;
  
  svg {
    color: var(--primary-color);
  }
`;

const SubcategoryGrid = styled.div`
  display: grid;
  grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
  gap: 1.5rem;
  margin-bottom: 2rem;
`;

const SubcategoryCard = styled(Link)`
  display: flex;
  flex-direction: column;
  padding: 1.5rem;
  background-color: var(--bg-color);
  border: 1px solid var(--border-color);
  border-radius: var(--radius);
  box-shadow: var(--box-shadow);
  transition: all var(--transition-speed);
  color: var(--text-color);
  
  &:hover {
    transform: translateY(-5px);
    box-shadow: var(--box-shadow-md);
    border-color: var(--primary-color);
  }
`;

const SubcategoryIcon = styled.div`
  width: 50px;
  height: 50px;
  background-color: rgba(79, 70, 229, 0.1);
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
  margin-bottom: 1rem;
  color: var(--primary-color);
  font-size: 1.5rem;
`;

const SubcategoryName = styled.h3`
  font-size: 1.25rem;
  margin-bottom: 0.5rem;
`;

const SubcategoryCount = styled.p`
  font-size: 0.9rem;
  color: var(--text-light);
  margin-bottom: 0.5rem;
`;

const SubcategoryDescription = styled.p`
  font-size: 0.9rem;
  color: var(--text-light);
  margin-bottom: 1rem;
  flex: 1;
`;

const TutorialSection = styled.section`
  margin-bottom: 2rem;
`;

const TutorialHeader = styled.div`
  display: flex;
  justify-content: space-between;
  align-items: center;
  margin-bottom: 1.5rem;
`;

const ViewOptions = styled.div`
  display: flex;
  gap: 0.5rem;
`;

const ViewButton = styled.button`
  background-color: ${props => props.active ? 'var(--primary-color)' : 'var(--bg-secondary)'};
  color: ${props => props.active ? 'white' : 'var(--text-color)'};
  border: none;
  border-radius: var(--radius);
  padding: 0.5rem;
  cursor: pointer;
  transition: all var(--transition-speed);
  display: flex;
  align-items: center;
  justify-content: center;
  
  &:hover {
    background-color: ${props => props.active ? 'var(--primary-hover)' : 'var(--border-color)'};
  }
`;

const TutorialGrid = styled.div`
  display: grid;
  grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
  gap: 1.5rem;
  margin-bottom: 1.5rem;
`;

const TutorialList = styled.div`
  display: flex;
  flex-direction: column;
  gap: 1rem;
  margin-bottom: 1.5rem;
`;

const TutorialCard = styled(Link)`
  display: flex;
  flex-direction: column;
  background-color: var(--bg-color);
  border: 1px solid var(--border-color);
  border-radius: var(--radius);
  overflow: hidden;
  box-shadow: var(--box-shadow);
  transition: all var(--transition-speed);
  color: var(--text-color);
  
  &:hover {
    transform: translateY(-5px);
    box-shadow: var(--box-shadow-md);
    border-color: var(--primary-color);
  }
`;

const TutorialListItem = styled(Link)`
  display: flex;
  gap: 1.5rem;
  padding: 1.5rem;
  background-color: var(--bg-color);
  border: 1px solid var(--border-color);
  border-radius: var(--radius);
  box-shadow: var(--box-shadow);
  transition: all var(--transition-speed);
  color: var(--text-color);
  
  &:hover {
    transform: translateY(-3px);
    box-shadow: var(--box-shadow-md);
    border-color: var(--primary-color);
  }
  
  @media (max-width: 768px) {
    flex-direction: column;
    gap: 1rem;
  }
`;

const TutorialContent = styled.div`
  padding: 1.5rem;
  flex: 1;
  display: flex;
  flex-direction: column;
`;

const TutorialListContent = styled.div`
  flex: 1;
  display: flex;
  flex-direction: column;
`;

const TutorialTitle = styled.h3`
  font-size: 1.1rem;
  margin-bottom: 0.5rem;
`;

const TutorialDescription = styled.p`
  font-size: 0.9rem;
  color: var(--text-light);
  margin-bottom: 1rem;
  flex: 1;
`;

const TutorialMeta = styled.div`
  display: flex;
  align-items: center;
  gap: 1rem;
  font-size: 0.8rem;
  color: var(--text-light);
`;

const TutorialMetaItem = styled.div`
  display: flex;
  align-items: center;
  gap: 0.25rem;
`;

const DifficultyBadge = styled.span`
  display: inline-block;
  padding: 0.25rem 0.5rem;
  border-radius: 12px;
  font-size: 0.75rem;
  font-weight: 600;
  text-transform: capitalize;
  background-color: ${props => {
    switch (props.level) {
      case 'beginner':
        return 'rgba(16, 185, 129, 0.1)';
      case 'intermediate':
        return 'rgba(249, 115, 22, 0.1)';
      case 'advanced':
        return 'rgba(239, 68, 68, 0.1)';
      default:
        return 'rgba(16, 185, 129, 0.1)';
    }
  }};
  color: ${props => {
    switch (props.level) {
      case 'beginner':
        return '#10b981';
      case 'intermediate':
        return '#f97316';
      case 'advanced':
        return '#ef4444';
      default:
        return '#10b981';
    }
  }};
  margin-bottom: 1rem;
`;

const EmptyState = styled.div`
  text-align: center;
  padding: 3rem 1rem;
  background-color: var(--bg-secondary);
  border-radius: var(--radius);
  
  h3 {
    margin-bottom: 1rem;
  }
  
  p {
    color: var(--text-light);
    margin-bottom: 1.5rem;
  }
`;

const LoadingState = styled.div`
  display: flex;
  justify-content: center;
  align-items: center;
  padding: 3rem 0;
  
  .spinner {
    width: 40px;
    height: 40px;
    border: 3px solid rgba(0, 0, 0, 0.1);
    border-radius: 50%;
    border-top-color: var(--primary-color);
    animation: spin 1s ease-in-out infinite;
  }
  
  @keyframes spin {
    to { transform: rotate(360deg); }
  }
`;

const CategoryPage = () => {
  const { slug } = useParams();
  const [category, setCategory] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [viewMode, setViewMode] = useState('grid');
  
  useEffect(() => {
    const fetchCategoryData = async () => {
      setLoading(true);
      try {
        const response = await categoriesAPI.getCategoryBySlug(slug);
        if (response.status === 'success') {
          setCategory(response.data);
        } else {
          setError('Failed to load category data');
        }
      } catch (err) {
        console.error('Error fetching category:', err);
        setError('Failed to load category. Please try again later.');
      } finally {
        setLoading(false);
      }
    };
    
    fetchCategoryData();
  }, [slug]);
  
  // Helper function to get appropriate icon for a subcategory
  const getSubcategoryIcon = (subcategoryName) => {
    const name = subcategoryName.toLowerCase();
    if (name.includes('frontend') || name.includes('react')) {
      return <FiBookOpen />;
    } else if (name.includes('backend') || name.includes('node')) {
      return <FiBarChart />;
    } else {
      return <FiBookOpen />;
    }
  };
  
  if (loading) {
    return (
      <CategoryContainer>
        <LoadingState>
          <div className="spinner"></div>
        </LoadingState>
      </CategoryContainer>
    );
  }
  
  if (error || !category) {
    return (
      <CategoryContainer>
        <EmptyState>
          <h3>Error Loading Category</h3>
          <p>{error || 'Category not found'}</p>
          <Link to="/" className="btn btn-primary">
            Return to Home
          </Link>
        </EmptyState>
      </CategoryContainer>
    );
  }
  
  return (
    <CategoryContainer>
      <CategoryHeader>
        <CategoryTitle>{category.name}</CategoryTitle>
        <CategoryDescription>
          {category.description || `Explore ${category.name} tutorials and guides.`}
        </CategoryDescription>
        <CategoryStats>
          <CategoryStat>
            <FiBookOpen />
            {category.tutorials?.length || 0} Tutorials
          </CategoryStat>
          {category.subcategories?.length > 0 && (
            <CategoryStat>
              <FiGrid />
              {category.subcategories.length} Subcategories
            </CategoryStat>
          )}
        </CategoryStats>
      </CategoryHeader>
      
      {category.subcategories?.length > 0 && (
        <SubcategorySection>
          <SectionTitle>
            <FiGrid /> Subcategories
          </SectionTitle>
          <SubcategoryGrid>
            {category.subcategories.map(subcategory => (
              <SubcategoryCard key={subcategory.id} to={`/category/${subcategory.slug}`}>
                <SubcategoryIcon>
                  {getSubcategoryIcon(subcategory.name)}
                </SubcategoryIcon>
                <SubcategoryName>{subcategory.name}</SubcategoryName>
                <SubcategoryDescription>
                  {subcategory.description || `Explore ${subcategory.name} tutorials and guides.`}
                </SubcategoryDescription>
              </SubcategoryCard>
            ))}
          </SubcategoryGrid>
        </SubcategorySection>
      )}
      
      <TutorialSection>
        <TutorialHeader>
          <SectionTitle>
            <FiBookOpen /> Tutorials
          </SectionTitle>
          <ViewOptions>
            <ViewButton 
              active={viewMode === 'grid'} 
              onClick={() => setViewMode('grid')}
              title="Grid View"
            >
              <FiGrid />
            </ViewButton>
            <ViewButton 
              active={viewMode === 'list'} 
              onClick={() => setViewMode('list')}
              title="List View"
            >
              <FiList />
            </ViewButton>
          </ViewOptions>
        </TutorialHeader>
        
        {category.tutorials?.length > 0 ? (
          viewMode === 'grid' ? (
            <TutorialGrid>
              {category.tutorials.map(tutorial => (
                <TutorialCard key={tutorial.id} to={`/tutorial/${tutorial.slug}`}>
                  <TutorialContent>
                    <DifficultyBadge level={tutorial.difficulty_level}>
                      {tutorial.difficulty_level}
                    </DifficultyBadge>
                    <TutorialTitle>{tutorial.title}</TutorialTitle>
                    <TutorialDescription>
                      {tutorial.description}
                    </TutorialDescription>
                    <TutorialMeta>
                      <TutorialMetaItem>
                        <FiClock /> {tutorial.estimated_time} min
                      </TutorialMetaItem>
                      <TutorialMetaItem>
                        <FiBarChart /> {tutorial.views} views
                      </TutorialMetaItem>
                    </TutorialMeta>
                  </TutorialContent>
                </TutorialCard>
              ))}
            </TutorialGrid>
          ) : (
            <TutorialList>
              {category.tutorials.map(tutorial => (
                <TutorialListItem key={tutorial.id} to={`/tutorial/${tutorial.slug}`}>
                  <TutorialListContent>
                    <DifficultyBadge level={tutorial.difficulty_level}>
                      {tutorial.difficulty_level}
                    </DifficultyBadge>
                    <TutorialTitle>{tutorial.title}</TutorialTitle>
                    <TutorialDescription>
                      {tutorial.description}
                    </TutorialDescription>
                    <TutorialMeta>
                      <TutorialMetaItem>
                        <FiClock /> {tutorial.estimated_time} min
                      </TutorialMetaItem>
                      <TutorialMetaItem>
                        <FiBarChart /> {tutorial.views} views
                      </TutorialMetaItem>
                    </TutorialMeta>
                  </TutorialListContent>
                </TutorialListItem>
              ))}
            </TutorialList>
          )
        ) : (
          <EmptyState>
            <h3>No Tutorials Found</h3>
            <p>There are no tutorials in this category yet.</p>
            <Link to="/" className="btn btn-primary">
              Browse Other Categories
            </Link>
          </EmptyState>
        )}
      </TutorialSection>
    </CategoryContainer>
  );
};

export default CategoryPage;