import React, { createContext, useState, useEffect, useContext } from 'react';

const UserProgressContext = createContext();

export const useUserProgress = () => useContext(UserProgressContext);

export const UserProgressProvider = ({ children }) => {
  // State for completed steps, saved tutorials, and recently viewed
  const [completedSteps, setCompletedSteps] = useState({});
  const [savedTutorials, setSavedTutorials] = useState([]);
  const [recentlyViewed, setRecentlyViewed] = useState([]);

  // Load user progress data from localStorage on mount
  useEffect(() => {
    const loadedCompletedSteps = localStorage.getItem('completedSteps');
    const loadedSavedTutorials = localStorage.getItem('savedTutorials');
    const loadedRecentlyViewed = localStorage.getItem('recentlyViewed');
    
    if (loadedCompletedSteps) {
      setCompletedSteps(JSON.parse(loadedCompletedSteps));
    }
    
    if (loadedSavedTutorials) {
      setSavedTutorials(JSON.parse(loadedSavedTutorials));
    }
    
    if (loadedRecentlyViewed) {
      setRecentlyViewed(JSON.parse(loadedRecentlyViewed));
    }
  }, []);

  // Save to localStorage whenever state changes
  useEffect(() => {
    localStorage.setItem('completedSteps', JSON.stringify(completedSteps));
  }, [completedSteps]);

  useEffect(() => {
    localStorage.setItem('savedTutorials', JSON.stringify(savedTutorials));
  }, [savedTutorials]);

  useEffect(() => {
    localStorage.setItem('recentlyViewed', JSON.stringify(recentlyViewed));
  }, [recentlyViewed]);

  // Toggle step completion
  const toggleStepCompletion = (tutorialId, stepId) => {
    setCompletedSteps(prev => {
      const tutorialSteps = prev[tutorialId] || [];
      const isCompleted = tutorialSteps.includes(stepId);
      
      if (isCompleted) {
        // Remove from completed steps
        return {
          ...prev,
          [tutorialId]: tutorialSteps.filter(id => id !== stepId)
        };
      } else {
        // Add to completed steps
        return {
          ...prev,
          [tutorialId]: [...tutorialSteps, stepId]
        };
      }
    });
  };

  // Check if a step is completed
  const isStepCompleted = (tutorialId, stepId) => {
    const tutorialSteps = completedSteps[tutorialId] || [];
    return tutorialSteps.includes(stepId);
  };

  // Calculate tutorial completion percentage
  const getTutorialProgress = (tutorialId, totalSteps) => {
    const tutorialSteps = completedSteps[tutorialId] || [];
    if (totalSteps === 0) return 0;
    return Math.round((tutorialSteps.length / totalSteps) * 100);
  };

  // Toggle saved tutorial
  const toggleSavedTutorial = (tutorial) => {
    setSavedTutorials(prev => {
      const isSaved = prev.some(item => item.id === tutorial.id);
      
      if (isSaved) {
        // Remove from saved tutorials
        return prev.filter(item => item.id !== tutorial.id);
      } else {
        // Add to saved tutorials
        return [...prev, tutorial];
      }
    });
  };

  // Check if a tutorial is saved
  const isTutorialSaved = (tutorialId) => {
    return savedTutorials.some(tutorial => tutorial.id === tutorialId);
  };

  // Add to recently viewed
  const addToRecentlyViewed = (tutorial) => {
    setRecentlyViewed(prev => {
      // Remove if already in the list
      const filtered = prev.filter(item => item.id !== tutorial.id);
      
      // Add to the beginning, limit to 10 items
      return [tutorial, ...filtered].slice(0, 10);
    });
  };

  return (
    <UserProgressContext.Provider value={{
      completedSteps,
      savedTutorials,
      recentlyViewed,
      toggleStepCompletion,
      isStepCompleted,
      getTutorialProgress,
      toggleSavedTutorial,
      isTutorialSaved,
      addToRecentlyViewed
    }}>
      {children}
    </UserProgressContext.Provider>
  );
};