import React, { useState, useEffect } from 'react';
import { Link, useLocation } from 'react-router-dom';
import styled from 'styled-components';
import { FiChevronDown, FiChevronRight, FiBookmark, FiClock, FiZap } from 'react-icons/fi';
import { categoriesAPI } from '../../services/api';
import { useUserProgress } from '../../contexts/UserProgressContext';

// Styled Components
const SidebarContainer = styled.aside`
  position: fixed;
  top: var(--header-height);
  left: 0;
  bottom: 0;
  width: var(--sidebar-width);
  background-color: var(--bg-secondary);
  border-right: 1px solid var(--border-color);
  overflow-y: auto;
  transform: translateX(${props => props.isOpen ? '0' : '-100%'});
  transition: transform var(--transition-speed);
  z-index: 90;
  
  @media (max-width: 768px) {
    box-shadow: var(--box-shadow-lg);
  }
`;

const SidebarSection = styled.div`
  margin-bottom: 1rem;
  
  &:not(:first-child) {
    border-top: 1px solid var(--border-color);
    padding-top: 1rem;
  }
`;

const SidebarSectionTitle = styled.h3`
  font-size: 0.85rem;
  text-transform: uppercase;
  color: var(--text-light);
  padding: 0.5rem 1.5rem;
  margin: 0;
`;

const CategoryList = styled.ul`
  list-style: none;
  padding: 0;
  margin: 0;
`;

const CategoryItem = styled.li`
  border-left: 3px solid ${props => props.active ? 'var(--primary-color)' : 'transparent'};
`;

const CategoryLink = styled(Link)`
  display: flex;
  align-items: center;
  justify-content: space-between;
  padding: 0.75rem 1.5rem;
  color: ${props => props.active ? 'var(--primary-color)' : 'var(--text-color)'};
  font-weight: ${props => props.active ? '600' : '400'};
  background-color: ${props => props.active ? 'rgba(79, 70, 229, 0.1)' : 'transparent'};
  transition: all var(--transition-speed);
  
  &:hover {
    background-color: ${props => props.active ? 'rgba(79, 70, 229, 0.15)' : 'rgba(0, 0, 0, 0.05)'};
  }
  
  .category-name {
    display: flex;
    align-items: center;
    
    .icon {
      margin-right: 0.5rem;
    }
    
    .count {
      margin-left: 0.5rem;
      font-size: 0.75rem;
      color: var(--text-light);
      background-color: var(--bg-color);
      border-radius: 10px;
      padding: 0.1rem 0.5rem;
    }
  }
`;

const SubcategoryList = styled.ul`
  list-style: none;
  padding: 0;
  margin: 0;
  overflow: hidden;
  max-height: ${props => props.isOpen ? `${props.itemCount * 40}px` : '0'};
  transition: max-height var(--transition-speed);
`;

const SubcategoryItem = styled.li`
  margin-left: 1.5rem;
  border-left: 1px solid var(--border-color);
`;

const SubcategoryLink = styled(Link)`
  display: block;
  padding: 0.5rem 1rem;
  color: ${props => props.active ? 'var(--primary-color)' : 'var(--text-color)'};
  font-weight: ${props => props.active ? '600' : '400'};
  font-size: 0.9rem;
  transition: all var(--transition-speed);
  
  &:hover {
    color: var(--primary-color);
  }
  
  &::before {
    content: '';
    display: inline-block;
    width: 5px;
    height: 5px;
    border-radius: 50%;
    background-color: ${props => props.active ? 'var(--primary-color)' : 'var(--text-light)'};
    margin-right: 0.5rem;
    vertical-align: middle;
  }
`;

const RecentItem = styled(Link)`
  display: flex;
  align-items: center;
  padding: 0.75rem 1.5rem;
  color: var(--text-color);
  transition: all var(--transition-speed);
  border-left: 3px solid transparent;
  
  &:hover {
    background-color: rgba(0, 0, 0, 0.05);
    color: var(--primary-color);
  }
  
  .icon {
    margin-right: 0.75rem;
    color: var(--text-light);
  }
  
  .content {
    overflow: hidden;
    
    .title {
      font-size: 0.9rem;
      margin: 0;
      white-space: nowrap;
      overflow: hidden;
      text-overflow: ellipsis;
    }
    
    .category {
      font-size: 0.75rem;
      color: var(--text-light);
    }
  }
`;

const ProgressIndicator = styled.div`
  width: 100%;
  height: 3px;
  background-color: var(--border-color);
  margin-top: 0.25rem;
  border-radius: 2px;
  overflow: hidden;
  
  .progress-bar {
    height: 100%;
    background-color: var(--secondary-color);
    width: ${props => props.progress}%;
  }
`;

const Sidebar = ({ isOpen }) => {
  const [categories, setCategories] = useState([]);
  const [expandedCategories, setExpandedCategories] = useState({});
  const [loading, setLoading] = useState(true);
  const location = useLocation();
  const { recentlyViewed, savedTutorials, getTutorialProgress } = useUserProgress();
  
  // Fetch categories on mount
  useEffect(() => {
    const fetchCategories = async () => {
      try {
        const response = await categoriesAPI.getAllCategories();
        if (response.status === 'success' && response.data) {
          setCategories(response.data);
          
          // Auto-expand categories based on current path
          const currentPath = location.pathname;
          if (currentPath.includes('/category/')) {
            const slug = currentPath.split('/category/')[1];
            const category = response.data.find(cat => cat.slug === slug);
            if (category) {
              setExpandedCategories(prev => ({
                ...prev,
                [category.id]: true
              }));
            }
          }
        }
      } catch (error) {
        console.error('Error fetching categories:', error);
      } finally {
        setLoading(false);
      }
    };
    
    fetchCategories();
  }, [location.pathname]);
  
  // Toggle category expansion
  const toggleCategory = (categoryId) => {
    setExpandedCategories(prev => ({
      ...prev,
      [categoryId]: !prev[categoryId]
    }));
  };
  
  // Check if a category is active (current page)
  const isCategoryActive = (slug) => {
    return location.pathname === `/category/${slug}`;
  };
  
  return (
    <SidebarContainer isOpen={isOpen}>
      <SidebarSection>
        <SidebarSectionTitle>Quick Start</SidebarSectionTitle>
        <CategoryList>
          <CategoryItem>
            <CategoryLink to="/">
              <span className="category-name">
                <FiZap className="icon" />
                Home
              </span>
            </CategoryLink>
          </CategoryItem>
          <CategoryItem>
            <CategoryLink to="/search?q=beginner%20tutorials">
              <span className="category-name">
                <FiZap className="icon" />
                Beginner Tutorials
              </span>
            </CategoryLink>
          </CategoryItem>
        </CategoryList>
      </SidebarSection>
      
      {recentlyViewed.length > 0 && (
        <SidebarSection>
          <SidebarSectionTitle>Recently Viewed</SidebarSectionTitle>
          {recentlyViewed.slice(0, 5).map(tutorial => (
            <RecentItem key={tutorial.id} to={`/tutorial/${tutorial.slug}`}>
              <FiClock className="icon" />
              <div className="content">
                <h4 className="title">{tutorial.title}</h4>
                <div className="category">{tutorial.category_name}</div>
                <ProgressIndicator progress={getTutorialProgress(tutorial.id, tutorial.steps?.length || 0)}>
                  <div className="progress-bar"></div>
                </ProgressIndicator>
              </div>
            </RecentItem>
          ))}
        </SidebarSection>
      )}
      
      {savedTutorials.length > 0 && (
        <SidebarSection>
          <SidebarSectionTitle>Saved Tutorials</SidebarSectionTitle>
          {savedTutorials.slice(0, 5).map(tutorial => (
            <RecentItem key={tutorial.id} to={`/tutorial/${tutorial.slug}`}>
              <FiBookmark className="icon" />
              <div className="content">
                <h4 className="title">{tutorial.title}</h4>
                <div className="category">{tutorial.category_name}</div>
              </div>
            </RecentItem>
          ))}
        </SidebarSection>
      )}
      
      <SidebarSection>
        <SidebarSectionTitle>Categories</SidebarSectionTitle>
        {loading ? (
          <div className="loading">Loading categories...</div>
        ) : (
          <CategoryList>
            {categories.map(category => (
              <React.Fragment key={category.id}>
                <CategoryItem active={isCategoryActive(category.slug)}>
                  <CategoryLink 
                    to={`/category/${category.slug}`}
                    active={isCategoryActive(category.slug) ? 1 : 0}
                  >
                    <span className="category-name">
                      {category.name}
                      {category.tutorial_count > 0 && (
                        <span className="count">{category.tutorial_count}</span>
                      )}
                    </span>
                    
                    {category.has_subcategories && (
                      <button 
                        onClick={(e) => {
                          e.preventDefault();
                          toggleCategory(category.id);
                        }}
                      >
                        {expandedCategories[category.id] ? <FiChevronDown /> : <FiChevronRight />}
                      </button>
                    )}
                  </CategoryLink>
                </CategoryItem>
                
                {category.has_subcategories && (
                  <SubcategoryList 
                    isOpen={expandedCategories[category.id]}
                    itemCount={category.subcategories?.length || 0}
                  >
                    {category.subcategories?.map(subcat => (
                      <SubcategoryItem key={subcat.id}>
                        <SubcategoryLink 
                          to={`/category/${subcat.slug}`}
                          active={isCategoryActive(subcat.slug) ? 1 : 0}
                        >
                          {subcat.name}
                        </SubcategoryLink>
                      </SubcategoryItem>
                    ))}
                  </SubcategoryList>
                )}
              </React.Fragment>
            ))}
          </CategoryList>
        )}
      </SidebarSection>
    </SidebarContainer>
  );
};

export default Sidebar;