import React, { useState, useEffect, useRef } from 'react';
import { Link, useNavigate, useLocation } from 'react-router-dom';
import styled from 'styled-components';
import { FiMenu, FiX, FiSearch, FiSun, FiMoon } from 'react-icons/fi';
import { tutorialsAPI } from '../../services/api';
import { useTheme } from '../../contexts/ThemeContext';

// Styled Components
const HeaderContainer = styled.header`
  position: fixed;
  top: 0;
  left: 0;
  right: 0;
  height: var(--header-height);
  background-color: var(--bg-color);
  border-bottom: 1px solid var(--border-color);
  display: flex;
  align-items: center;
  padding: 0 1.5rem;
  z-index: 100;
  box-shadow: var(--box-shadow);
  transition: background-color var(--transition-speed), border-color var(--transition-speed);
`;

const Logo = styled(Link)`
  font-size: 1.5rem;
  font-weight: 700;
  color: var(--primary-color);
  display: flex;
  align-items: center;
  margin-right: 2rem;
  
  span {
    margin-left: 0.5rem;
  }
  
  @media (max-width: 768px) {
    font-size: 1.25rem;
  }
`;

const MenuButton = styled.button`
  background: none;
  border: none;
  color: var(--text-color);
  font-size: 1.5rem;
  display: flex;
  align-items: center;
  justify-content: center;
  cursor: pointer;
  margin-right: 1rem;
  transition: color var(--transition-speed);
  
  &:hover {
    color: var(--primary-color);
  }
`;

const SearchContainer = styled.div`
  position: relative;
  flex: 1;
  max-width: 600px;
  margin: 0 1rem;
  
  @media (max-width: 768px) {
    max-width: none;
  }
`;

const SearchForm = styled.form`
  display: flex;
  width: 100%;
`;

const SearchInput = styled.input`
  width: 100%;
  padding: 0.5rem 1rem 0.5rem 2.5rem;
  border: 1px solid var(--border-color);
  border-radius: var(--radius);
  background-color: var(--bg-secondary);
  color: var(--text-color);
  font-size: 0.9rem;
  transition: all var(--transition-speed);
  
  &:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 2px rgba(79, 70, 229, 0.2);
  }
`;

const SearchIcon = styled.div`
  position: absolute;
  left: 0.75rem;
  top: 50%;
  transform: translateY(-50%);
  color: var(--text-light);
  pointer-events: none;
`;

const SearchResults = styled.div`
  position: absolute;
  top: 100%;
  left: 0;
  right: 0;
  background-color: var(--bg-color);
  border: 1px solid var(--border-color);
  border-radius: var(--radius);
  margin-top: 0.5rem;
  box-shadow: var(--box-shadow-md);
  max-height: 400px;
  overflow-y: auto;
  z-index: 10;
  display: ${props => props.show ? 'block' : 'none'};
`;

const SearchResultItem = styled(Link)`
  display: block;
  padding: 0.75rem 1rem;
  border-bottom: 1px solid var(--border-color);
  color: var(--text-color);
  transition: background-color var(--transition-speed);
  
  &:last-child {
    border-bottom: none;
  }
  
  &:hover {
    background-color: var(--bg-secondary);
  }
  
  h4 {
    margin: 0 0 0.25rem;
    font-size: 0.95rem;
  }
  
  p {
    margin: 0;
    font-size: 0.85rem;
    color: var(--text-light);
  }
`;

const Breadcrumbs = styled.div`
  display: flex;
  align-items: center;
  margin-left: 1rem;
  font-size: 0.85rem;
  color: var(--text-light);
  
  @media (max-width: 768px) {
    display: none;
  }
`;

const BreadcrumbItem = styled(Link)`
  color: var(--text-light);
  
  &:hover {
    color: var(--primary-color);
  }
  
  &:not(:last-child)::after {
    content: '/';
    margin: 0 0.5rem;
    color: var(--text-light);
  }
`;

const CurrentBreadcrumb = styled.span`
  color: var(--text-color);
  font-weight: 500;
`;

const RightActions = styled.div`
  display: flex;
  align-items: center;
  margin-left: auto;
`;

const ThemeToggle = styled.button`
  background: none;
  border: none;
  color: var(--text-color);
  font-size: 1.25rem;
  display: flex;
  align-items: center;
  justify-content: center;
  cursor: pointer;
  margin-left: 1rem;
  transition: color var(--transition-speed);
  
  &:hover {
    color: var(--primary-color);
  }
`;

const Header = ({ toggleSidebar, sidebarOpen }) => {
  const [searchQuery, setSearchQuery] = useState('');
  const [searchResults, setSearchResults] = useState([]);
  const [showResults, setShowResults] = useState(false);
  const [isSearching, setIsSearching] = useState(false);
  const searchRef = useRef(null);
  const navigate = useNavigate();
  const location = useLocation();
  const { darkMode, toggleTheme } = useTheme();
  
  // Close search results when clicking outside
  useEffect(() => {
    const handleClickOutside = (event) => {
      if (searchRef.current && !searchRef.current.contains(event.target)) {
        setShowResults(false);
      }
    };
    
    document.addEventListener('mousedown', handleClickOutside);
    return () => {
      document.removeEventListener('mousedown', handleClickOutside);
    };
  }, []);
  
  // Reset search when location changes
  useEffect(() => {
    setSearchQuery('');
    setShowResults(false);
  }, [location]);
  
  // Handle search input change without triggering search
  const handleSearchChange = (e) => {
    setSearchQuery(e.target.value);
  };
  
  // Search tutorials API call
  const searchTutorials = async (query) => {
    if (!query.trim() || query.length < 2) return;
    
    setIsSearching(true);
    
    try {
      const response = await tutorialsAPI.searchTutorials(query);
      if (response.results && response.results.length > 0) {
        setSearchResults(response.results);
        setShowResults(true);
      } else {
        setSearchResults([]);
      }
    } catch (error) {
      console.error('Error searching tutorials:', error);
      setSearchResults([]);
    } finally {
      setIsSearching(false);
    }
  };
  
  // Handle search form submission
  const handleSearchSubmit = (e) => {
    e.preventDefault();
    if (searchQuery.trim()) {
      // For instant search results in dropdown
      searchTutorials(searchQuery);
      
      // For full search page navigation
      if (searchQuery.trim().length >= 2) {
        navigate(`/search?q=${encodeURIComponent(searchQuery.trim())}`);
        setShowResults(false);
      }
    }
  };
  
  // Generate breadcrumbs based on current location
  const renderBreadcrumbs = () => {
    const pathnames = location.pathname.split('/').filter(x => x);
    
    if (pathnames.length === 0) {
      return null;
    }
    
    return (
      <Breadcrumbs>
        <BreadcrumbItem to="/">Home</BreadcrumbItem>
        
        {pathnames.map((value, index) => {
          const isLast = index === pathnames.length - 1;
          const path = `/${pathnames.slice(0, index + 1).join('/')}`;
          
          // Format the breadcrumb text
          let breadcrumbText = value.charAt(0).toUpperCase() + value.slice(1);
          breadcrumbText = breadcrumbText.replace(/-/g, ' ');
          
          return isLast ? (
            <CurrentBreadcrumb key={path}>{breadcrumbText}</CurrentBreadcrumb>
          ) : (
            <BreadcrumbItem key={path} to={path}>
              {breadcrumbText}
            </BreadcrumbItem>
          );
        })}
      </Breadcrumbs>
    );
  };
  
  return (
    <HeaderContainer>
      <MenuButton onClick={toggleSidebar}>
        {sidebarOpen ? <FiX /> : <FiMenu />}
      </MenuButton>
      
      <Logo to="/">
        <span>TutorAI</span>
      </Logo>
      
      {renderBreadcrumbs()}
      
      <SearchContainer ref={searchRef}>
        <SearchForm onSubmit={handleSearchSubmit}>
          <SearchIcon>
            <FiSearch />
          </SearchIcon>
          <SearchInput
            type="text"
            placeholder="Search tutorials..."
            value={searchQuery}
            onChange={handleSearchChange}
            onKeyDown={(e) => {
              if (e.key === 'Enter') {
                e.preventDefault();
                handleSearchSubmit(e);
              }
            }}
          />
        </SearchForm>
        
        <SearchResults show={showResults && searchResults.length > 0}>
          {searchResults.map(tutorial => (
            <SearchResultItem 
              key={tutorial.id} 
              to={`/tutorial/${tutorial.slug || tutorial.tutorial?.slug}`}
              onClick={() => setShowResults(false)}
            >
              <h4>{tutorial.title}</h4>
              <p>{tutorial.category_name} • {tutorial.difficulty_level}</p>
            </SearchResultItem>
          ))}
        </SearchResults>
      </SearchContainer>
      
      <RightActions>
        <ThemeToggle onClick={toggleTheme}>
          {darkMode ? <FiSun /> : <FiMoon />}
        </ThemeToggle>
      </RightActions>
    </HeaderContainer>
  );
};

export default Header;