import React, { useState, useEffect } from "react";
import { Link } from "react-router-dom";
import styled from "styled-components";
import {
  FiThumbsUp,
  FiThumbsDown,
  FiPrinter,
  FiDownload,
} from "react-icons/fi";

// Styled Components
const FooterContainer = styled.footer`
  background-color: var(--bg-secondary);
  border-top: 1px solid var(--border-color);
  padding: 2rem 0;
  color: var(--text-color);
  transition: background-color var(--transition-speed),
    border-color var(--transition-speed),
    margin-left var(--transition-speed);
  
  /* Apply margin-left conditionally based on sidebar state */
  margin-left: ${props => props.sidebarOpen ? 'var(--sidebar-width)' : '0'};
  
  @media (max-width: 768px) {
    margin-left: 0;
  }
`;

const FooterContent = styled.div`
  display: flex;
  flex-wrap: wrap;
  justify-content: space-between;
  max-width: 1200px;
  margin: 0 auto;
  padding: 0 1.5rem;

  @media (max-width: 768px) {
    flex-direction: column;
    gap: 2rem;
  }
`;

const FooterSection = styled.div`
  flex: 1;
  min-width: 200px;
  margin-right: 2rem;

  @media (max-width: 768px) {
    margin-right: 0;
  }
`;

const FooterTitle = styled.h3`
  font-size: 1rem;
  margin-bottom: 1rem;
  color: var(--text-color);
`;

const FooterList = styled.ul`
  list-style: none;
  padding: 0;
  margin: 0;
`;

const FooterListItem = styled.li`
  margin-bottom: 0.5rem;
`;

const FooterLink = styled(Link)`
  color: var(--text-light);
  font-size: 0.9rem;
  transition: color var(--transition-speed);

  &:hover {
    color: var(--primary-color);
  }
`;

const FeedbackSection = styled.div`
  display: flex;
  flex-direction: column;
  align-items: flex-start;
`;

const FeedbackQuestion = styled.p`
  margin-bottom: 1rem;
  font-size: 0.9rem;
`;

const FeedbackButtons = styled.div`
  display: flex;
  gap: 1rem;
`;

const FeedbackButton = styled.button`
  display: flex;
  align-items: center;
  gap: 0.5rem;
  padding: 0.5rem 1rem;
  background-color: ${(props) =>
    props.active ? "var(--primary-color)" : "transparent"};
  color: ${(props) => (props.active ? "white" : "var(--text-color)")};
  border: 1px solid
    ${(props) =>
      props.active ? "var(--primary-color)" : "var(--border-color)"};
  border-radius: var(--radius);
  font-size: 0.9rem;
  cursor: pointer;
  transition: all var(--transition-speed);

  &:hover {
    background-color: ${(props) =>
      props.active ? "var(--primary-hover)" : "rgba(0, 0, 0, 0.05)"};
    border-color: ${(props) =>
      props.active ? "var(--primary-hover)" : "var(--text-light)"};
  }
`;

const FooterBottom = styled.div`
  display: flex;
  justify-content: space-between;
  align-items: center;
  max-width: 1200px;
  margin: 2rem auto 0;
  padding: 1.5rem 1.5rem 0;
  border-top: 1px solid var(--border-color);
  font-size: 0.85rem;
  color: var(--text-light);

  @media (max-width: 768px) {
    flex-direction: column;
    gap: 1rem;
    text-align: center;
  }
`;

const FooterActions = styled.div`
  display: flex;
  gap: 1rem;
`;

const ActionButton = styled.button`
  display: flex;
  align-items: center;
  gap: 0.5rem;
  background: none;
  border: none;
  color: var(--text-light);
  font-size: 0.85rem;
  cursor: pointer;
  transition: color var(--transition-speed);

  &:hover {
    color: var(--primary-color);
  }
`;

const LoadingText = styled.p`
  font-size: 0.9rem;
  color: var(--text-light);
  font-style: italic;
`;

const Footer = ({ sidebarOpen }) => {
  const [feedback, setFeedback] = useState(null);
  const [popularTutorials, setPopularTutorials] = useState([]);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    // Fetch popular tutorials
    const fetchPopularTutorials = async () => {
      try {
        const response = await fetch('/api/tutorials?sort_by=views&sort_order=desc&limit=5');
        if (!response.ok) {
          throw new Error('Failed to fetch popular tutorials');
        }
        const data = await response.json();
        setPopularTutorials(data.data || []);
      } catch (error) {
        console.error('Error fetching popular tutorials:', error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchPopularTutorials();
  }, []);

  const handleFeedback = (value) => {
    setFeedback(value);
    // Here you would typically send the feedback to your backend
    console.log(`User feedback: ${value}`);
  };

  const handlePrint = () => {
    window.print();
  };

  return (
    <FooterContainer className="no-print" sidebarOpen={sidebarOpen}>
      <FooterContent>
        <FooterSection>
          <FooterTitle>Popular Tutorials</FooterTitle>
          {isLoading ? (
            <LoadingText>Loading popular tutorials...</LoadingText>
          ) : popularTutorials.length > 0 ? (
            <FooterList>
              {popularTutorials.map((tutorial) => (
                <FooterListItem key={tutorial.id}>
                  <FooterLink to={`/tutorial/${tutorial.slug}`}>
                    {tutorial.title}
                  </FooterLink>
                </FooterListItem>
              ))}
            </FooterList>
          ) : (
            <LoadingText>No tutorials found</LoadingText>
          )}
        </FooterSection>

        <FooterSection>
          <FeedbackSection>
            <FooterTitle>Feedback</FooterTitle>
            <FeedbackQuestion>Was this documentation helpful?</FeedbackQuestion>
            <FeedbackButtons>
              <FeedbackButton
                active={feedback === "helpful"}
                onClick={() => handleFeedback("helpful")}>
                <FiThumbsUp /> Yes
              </FeedbackButton>
              <FeedbackButton
                active={feedback === "not-helpful"}
                onClick={() => handleFeedback("not-helpful")}>
                <FiThumbsDown /> No
              </FeedbackButton>
            </FeedbackButtons>
          </FeedbackSection>
        </FooterSection>
      </FooterContent>

      <FooterBottom>
        <div>
          <p>© {new Date().getFullYear()} TutorAI. All rights reserved.</p>
          <p>Last updated: {new Date().toLocaleDateString()}</p>
        </div>

        <FooterActions>
          <ActionButton onClick={handlePrint}>
            <FiPrinter /> Print-friendly version
          </ActionButton>
        </FooterActions>
      </FooterBottom>
    </FooterContainer>
  );
};

export default Footer;